using System;
using System.Diagnostics.Contracts;

namespace CaseStudy
{

public class Vector {
  
	private object[] elementData;
	public int elementCount;
	private int modCount;
	protected int capacityIncrement;
  
	[ContractInvariantMethod]
    [ContractVerification(false)]
	private void ObjectInvariant() {
		Contract.Invariant(elementData != null);
		Contract.Invariant(Contract.ForAll(elementCount, elementData.Length, i => elementData[i] == null));
		// Skipped owner condition for elementData.
		Contract.Invariant(0 <= elementCount && elementCount <= elementData.Length);
		Contract.Invariant(capacityIncrement >= 0);
	}
  
	// At least one constructor is required, to ensure that elementData != null.
    [ContractVerification(false)]
	public Vector(int initialCapacity) {

		if (initialCapacity < 0) {
			throw new ArgumentException ();
		}
		this.elementData = new object[initialCapacity];
	}

    [ContractVerification(false)]
	public void removeAllElements() {
		// Skip assignable annotation
		Contract.Ensures(elementCount == 0);
		Contract.Ensures(TLists.eq(elementData, 0, Contract.OldValue(elementCount), null));
		Contract.Ensures(TLists.eq(elementData, Contract.OldValue(elementCount), elementData.Length, null));

		modCount++;
		for (int ic = 0; ic < elementCount; ic++) {
			elementData[ic] = null;
		}
		elementCount = 0;
	}
	
	[Pure]
    [ContractVerification(false)]
	public int indexOf1(object value, int index) {
		Contract.Requires(0 <= index && index <= elementCount);
        // Equivalence was split up for readability, but is counted as one post-condition.
		Contract.Ensures(Contract.Result<int>() != -1 || !TLists.In(elementData, index, elementCount, value));
		Contract.Ensures(Contract.Result<int>() == -1 ||  TLists.In(elementData, index, elementCount, value));
		Contract.Ensures(Contract.Result<int>() == -1 || index <= Contract.Result<int>() && Contract.Result<int>() < elementCount);
		Contract.Ensures(Contract.Result<int>() == -1 || elementData[Contract.Result<int>()] == value && !TLists.In(elementData, index, Contract.Result<int>(), value));

		if (index < 0 || index > elementCount)
			throw new IndexOutOfRangeException();

		if (value == null) {
			for (int ic = index; ic < elementCount; ic++) {
				if (elementData[ic] == null) {
          			return ic;
        		}
      		}
		} else {
			for (int ic = index; ic < elementCount; ic++) {
				if (value == elementData[ic]) {
					return ic;
				}
			}
		}
		return -1;
	}
	
	[Pure]
    [ContractVerification(false)]
	public int lastIndexOf1(object value, int index) {
		Contract.Requires(0 <= index && index <= elementCount);
        // Equivalence was split up for readability, but is counted as one post-condition.
		Contract.Ensures(Contract.Result<int>() != -1 || !TLists.In(elementData, 0, index + 1, value));
		Contract.Ensures(Contract.Result<int>() == -1 ||  TLists.In(elementData, 0, index + 1, value));
		Contract.Ensures(Contract.Result<int>() == -1 || 0 <= Contract.Result<int>() && Contract.Result<int>() <= index);
		Contract.Ensures(Contract.Result<int>() == -1 || elementData[Contract.Result<int>()] == value && !TLists.In(elementData, Contract.Result<int>() + 1, index + 1, value));

		if (index < 0 || index >= elementCount) {
			throw new IndexOutOfRangeException ();
		}

		if (value == null) {
			for (int ic = index; ic >= 0; ic--) {
				if (elementData[ic] == null) {
					return ic;
				}
			}
		} else {
			for (int ic = index; ic >= 0; ic--) {
				if (value == elementData[ic]) {
					return ic;
				}
			}
		}
		return -1;
	}

    [ContractVerification(false)]
	public void setSize(int newSize) {
		Contract.Requires(newSize >= 0);
		// Skip assignable statement
		Contract.Ensures(elementCount == newSize);
		Contract.Ensures(Contract.ForAll(0, newSize <= Contract.OldValue(elementCount) ? newSize : Contract.OldValue(elementCount), i => elementData[i] == Contract.OldValue(elementData[i])));
		Contract.Ensures(newSize >= Contract.OldValue(elementCount) || TLists.eq(elementData, newSize, elementData.Length, null));
		if (newSize < 0)
			throw new IndexOutOfRangeException();

		modCount++;
		if (newSize > elementCount) {
			ensureCapacityHelper(newSize);
			Contract.Assume(Contract.ForAll(0, elementCount, i => elementData[i] == Contract.OldValue(elementData[i])));
			// Contract.Assume(TLists.eq(elementData, elementCount, newSize, null)); removed because emits warning
		} else {
			for (int ic = newSize; ic < elementCount; ic++) {
				elementData[ic] = null;
			}
		}
		elementCount = newSize;
	}

    [ContractVerification(false)]
	private void ensureCapacityHelper(int minCapacity) {
		// Skip assignable annotation
		Contract.Ensures(elementData.Length >= Contract.OldValue(elementData.Length));
		Contract.Ensures(elementData.Length >= minCapacity);
		int oldCapacity = elementData.Length;
		if (minCapacity > oldCapacity) {
			object[] oldData = elementData;
			int newCapacity = (capacityIncrement > 0) ? (oldCapacity + capacityIncrement) : (oldCapacity * 2);
			if (newCapacity < minCapacity) {
				newCapacity = minCapacity;
			}
			elementData = Arrays.copyOf(elementData, newCapacity);
		}
	}

    [ContractVerification(false)]
	public void removeRange0(int fromIndex, int toIndex) {
		Contract.Requires(0 <= fromIndex && fromIndex <= toIndex && toIndex < elementCount);
		// Skip assignable annotation
		Contract.Requires(fromIndex <= toIndex);
		Contract.Ensures(elementCount == Contract.OldValue(elementCount) - (toIndex - fromIndex));
		Contract.Ensures(Contract.ForAll(0, fromIndex, i => elementData[i] == Contract.OldValue(elementData[i])));
		Contract.Ensures(Contract.ForAll(0, Contract.OldValue(elementCount) - toIndex, i => elementData[fromIndex + i] == Contract.OldValue(elementData[toIndex + i])));
		Contract.Ensures(TLists.eq(elementData, elementCount, elementData.Length, null));
		modCount++;
		int numMoved = elementCount - toIndex;
		SystemUtil.arraycopyInteger(elementData, toIndex, elementData, fromIndex, numMoved);

		int oldElementCount = elementCount;
		int newElementCount = oldElementCount - (toIndex - fromIndex);
		while (elementCount != newElementCount) {
			elementData[--elementCount] = null;
		}
	}
  
}
}