using System;
using System.Diagnostics.Contracts;

namespace CaseStudy
{
public class Arrays
{

	[Pure]
    [ContractVerification(false)]
	public static int binarySearch(int[] a, int fromIndex, int toIndex, int key) {
		Contract.Requires(a != null);
		// Contract.Requires (TArrays.within(a, fromIndex, toIndex)); inlined for improved analysis
        Contract.Requires(0 <= fromIndex && fromIndex <= toIndex && toIndex <= a.Length);
		// Contract.Requires(TArrays.sorted(a, fromIndex, toIndex)); inlined for improved analysis
        Contract.Requires(Contract.ForAll(fromIndex, toIndex - 1, i => a[i] <= a[i + 1]));
		Contract.Ensures((Contract.Result<int>() < 0)  || (a[Contract.Result<int>()] == key));
		// Contract.Ensures((Contract.Result<int>() >= 0) || !TArrays.In(a, fromIndex, toIndex, key)); inlined for improved analysis
        Contract.Ensures((Contract.Result<int>() >= 0) || Contract.ForAll(fromIndex, toIndex, i => a[i] != key));
		int low = fromIndex;
		int high = toIndex -1;
		while (low <= high) {
			int mid = low + (high - low) / 2;
			int midVal = a[mid];
			if (midVal < key) {
				low = mid + 1;
			}
			else {
				if (midVal > key) {
					high = mid - 1;
				}
				else {
					return mid;
				}
			}
		}
		return -(low + 1);
	}

	// Skip binary search wrapper methods.

    [ContractVerification(false)]
	public static object[] copyOf(object[] original, int newLength) {
		Contract.Requires(original != null);
		Contract.Requires(newLength >= 0);
		//Skip assignable assignment.
		Contract.Ensures(Contract.Result<object[]>() != null);
		Contract.Ensures(Contract.ForAll (0, original.Length, i => original[i] == Contract.OldValue(original[i])));
		Contract.Ensures(Contract.ForAll (original.Length, newLength, i => Contract.Result<object[]>()[i] == null));
		Contract.Ensures(Contract.Result<object[]>().Length == newLength);
		Contract.Ensures(Contract.Result<object[]>().Length  > original.Length || Contract.ForAll(0, Contract.Result<object[]>().Length, i => Contract.Result<object[]>()[i] == original[i]));
		Contract.Ensures(Contract.Result<object[]>().Length <= original.Length || Contract.ForAll(0,                    original.Length, i => Contract.Result<object[]>()[i] == original[i]));
		object[] copy = new object[newLength];
		SystemUtil.arraycopyInteger (original, 0, copy, 0, Math.Min (original.Length, newLength));
		return copy;
	}

	// Skip copyOfRange

    [ContractVerification(false)]
	private static void fill1(int[] a, int fromIndex, int toIndex, int val) {
		Contract.Requires(a != null);
		// Contract.Requires(TArrays.within(a, fromIndex, toIndex)); inlined for better analysis
        Contract.Requires(0 <= fromIndex && fromIndex <= toIndex && toIndex <= a.Length);
		// Skip assignable annotation.
		Contract.Ensures(Contract.ForAll(0, fromIndex, i => a[i] == Contract.OldValue(a[i])));
		// Contract.Ensures(TArrays.eq(a, fromIndex, toIndex, val)); inlined for better analysis
        Contract.Ensures(Contract.ForAll(fromIndex, toIndex, i => a[i] == val));
		Contract.Ensures(Contract.ForAll(toIndex, a.Length, i => a[i] == Contract.OldValue(a[i])));
		rangeCheck(a.Length, fromIndex, toIndex);
		for (int ic = fromIndex; ic < toIndex; ic++) {
			a[ic] = val;
		}
	}

    [ContractVerification(false)]
	public static void fill0(int[] a, int val) {
		Contract.Requires(a !=  null);
		// Skip assignable annotation.
		Contract.Ensures(TArrays.eq(a, 0, a.Length, val));
		for (int ic = 0; ic < a.Length; ic++) {
			a[ic] = val;
		}
	}

    [ContractVerification(false)]
	public static void sortInteger(object[] a) {
		// Skip assignable annotation
		Contract.Ensures(Contract.ForAll(0, a.Length, i => a[i] != null));
		Contract.Ensures(Contract.ForAll(0, a.Length-1, i => a[i].GetHashCode() <= a[i+1].GetHashCode()));

		object[] aux = (object[]) a.Clone();
		mergeSort(aux, a, 0, a.Length, 0);
	}

    [ContractVerification(false)]
	private static PartitionIndexes quicksortPartition(int[] x, int off, int len, int v) {
		Contract.Requires(x != null);
		Contract.Requires(len >= 0);
		Contract.Requires(off >= 0);
		Contract.Requires(off + len <= x.Length);
		// Skip assignable annotation
		Contract.Ensures(Contract.ForAll(0, off, i => x[i] == Contract.OldValue(x[i])));
		Contract.Ensures(Contract.ForAll(off + len, x.Length, i => x[i] == Contract.OldValue(x[i])));
		Contract.Ensures(TArrays.eq(x, off, Contract.Result<PartitionIndexes>().a, v));
		Contract.Ensures(TArrays.less(x, Contract.Result<PartitionIndexes>().a, Contract.Result<PartitionIndexes>().b, v));
		Contract.Ensures(TArrays.grt(x, Contract.Result<PartitionIndexes>().c + 1, Contract.Result<PartitionIndexes>().d + 1, v));
		Contract.Ensures(TArrays.eq(x, Contract.Result<PartitionIndexes>().d + 1, off + len, v));
		Contract.Ensures(off <= Contract.Result<PartitionIndexes>().a);
		Contract.Ensures(Contract.Result<PartitionIndexes>().a <= Contract.Result<PartitionIndexes>().b);
		Contract.Ensures(Contract.Result<PartitionIndexes>().b == Contract.Result<PartitionIndexes>().c + 1);
		Contract.Ensures(Contract.Result<PartitionIndexes>().c <= Contract.Result<PartitionIndexes>().d);
		Contract.Ensures(Contract.Result<PartitionIndexes>().d < off + len);
		int a = off, b = a, c = off + len - 1, d = c;
		while (true) {
			while ( b <= c && x[b] <= v) {
				if (x[b] == v) {
					swap(x, a++, b);
				}
				b++;
			}
			while ( c >= b && x[c] >= v) {
				if (x[c] == v) {
					swap(x, c, d--);
				}
				c--;
			}
			if (b > c) {
				break;
			}
			swap(x, b++, c--); 
		}
		PartitionIndexes idxs = new PartitionIndexes();
		idxs.a = a;
		idxs.b = b;
		idxs.c = c;
		idxs.d = d;
		return idxs;
	}

    [ContractVerification(false)]
	private static void insertionSort1(int[] x, int off, int len) {
		Contract.Requires(x != null);
		Contract.Requires(TArrays.within(x, off, off + len));
		// Skip assignable annotation
		Contract.Ensures(TArrays.sorted(x, off, off + len));
		if (x == null) {
			throw new NullReferenceException();
		}
		if ( off < 0 || len > x.Length || len < 0 || off > x.Length || off + len > x.Length) {
			throw new Exception();
		}
		if (len == 0) {
			return;
		}
		if (len == 1) {
			return;
		}
		int high = off + len;
		int ic = off + 1;
		while ( ic < high ) {
			int newValue = x[ic];
			x[ic] = x[ic - 1];
			int jc = ic;
			while ( jc > off && x[jc - 1] > newValue) {
				x[jc] = x[jc - 1];
				jc--;
			}
			x[jc] = newValue;
			ic++;
		}
	}

    [ContractVerification(false)]
	private static void swap(int[] x, int a, int b) {
		Contract.Requires(x != null);
		// Contract.Requires(TArrays.in_bound(x, a) && TArrays.in_bound(x, b)); inlined for better analysis
        Contract.Requires((0 <= a && a < x.Length) && (0 <= b && b < x.Length));
		// Skip assignable annotation.
		Contract.Ensures(x[a] == Contract.OldValue(x[b]));
		Contract.Ensures(x[b] == Contract.OldValue(x[a]));
		Contract.Ensures(Contract.ForAll(0, x.Length, i => (i == a || i == b || x[i] == Contract.OldValue(x[i]))));
		int t = x[a];
		x[a] = x[b];
		x[b] = t;
	}

	// Skip med3

    [ContractVerification(false)]
	private static void vecswap(int[] x, int a, int b, int n) {
		Contract.Requires(x != null);
		Contract.Requires(n >= 0);
		// Contract.Requires(TArrays.within(x, a, a+n)); inlined for better analysis
        Contract.Requires(0 <= a && a <= a+n && a+n <= x.Length);
		// Contract.Requires(TArrays.within(x, b, b+n)); inlined for better analysis
        Contract.Requires(0 <= b && b <= b+n && b+n <= x.Length);
		Contract.Requires(a + n <= b || b + n <= a);
		// Skip assignable annotation
		Contract.Ensures(Contract.ForAll(0, a < b ? a : b, i => x[i] == Contract.OldValue(x[i])));
		Contract.Ensures(Contract.ForAll((a > b ? a : b) + n, x.Length, i => x[i] == Contract.OldValue(x[i])));
		Contract.Ensures(Contract.ForAll(0, n, i => x[a + i] == Contract.OldValue(x[b + i])));
		Contract.Ensures(Contract.ForAll(0, n, i => x[b + i] == Contract.OldValue(x[a + i])));
		for (int ic = 0; ic < n; ic++) {
			swap (x, a + ic, b + ic);
		}
	}

	[Pure]
    [ContractVerification(false)]
	private static void rangeCheck(int arrayLen, int fromIndex, int toIndex) {
		Contract.Requires(0 <= fromIndex && fromIndex <= toIndex && toIndex <= arrayLen);
        Contract.Requires<ArgumentException>(fromIndex <= toIndex);
        Contract.Requires<ArgumentOutOfRangeException>(fromIndex >= 0 || toIndex <= arrayLen);
		if (fromIndex > toIndex) {
			throw new ArgumentException();
		}
		if (fromIndex < 0) {
			throw new ArgumentOutOfRangeException();
		}
		if (toIndex > arrayLen) {
			throw new ArgumentOutOfRangeException();
		}
	}

	private static int INSERTIONSORT_THRESHOLD = 7;

    [ContractVerification(false)]
	private static void mergeSort(object[] src, object[] dest, int low, int high, int off) {
		int length = high - low;

		if (length < INSERTIONSORT_THRESHOLD) {
			insertionSort0(dest, low, high);
			return;
		}

		int destLow = low;
		int destHigh = high;
		low += off;
		high += off;
		int mid = MathAux.unsigned_right_shift(low + high, 1);
		mergeSort(dest, src, low, mid, -off);
		mergeSort(dest, src, mid, high, -off);

		if (src[mid - 1].GetHashCode() <= src[mid].GetHashCode()) {
			SystemUtil.arraycopyInteger(src, low, dest, destLow, length);
			return;
		}

			merge0(src, dest, low, mid, high, destLow, destHigh);
	}

    [ContractVerification(false)]	
	private static void merge0(object[] src, object[] dest, int low, int mid, int high, int destLow, int destHigh) {
		Contract.Requires(src != null && dest != null);
		Contract.Requires(TArrays.sorted(src, low, mid));
		Contract.Requires(TArrays.sorted(src, mid, high));
		//Contract.Requires(TArrays.within(src, low, high)); inlined for better analysis
        Contract.Requires(0 <= low && low <= high && high <= src.Length);
		Contract.Requires(low <= mid && mid <= high);
		//Contract.Requires(TArrays.within(dest, destLow, destHigh)); inlined for better analysis
        Contract.Requires(0 <= destLow && destLow <= destHigh && destHigh <= dest.Length);
		Contract.Requires(high - low == destHigh - destLow);
		Contract.Requires(TArrays.nonnull(src, low, high));
		Contract.Requires(src != dest || (high <= destLow || destHigh <= low));
		// Skip assignable annotation.
		Contract.Ensures(TArrays.sorted(dest, destLow, destHigh));
		int ic = destLow;
		int p = low;
		int q = mid;
		for( ; ic < destHigh; ic++) {
				if ( q >= high || p < mid && (src[p].GetHashCode() < src[q].GetHashCode())) {
				dest[ic] = src[p++];
			}
			else {
				dest[ic] = src[q++];
			}
		}
	}

    [ContractVerification(false)]
	private static void insertionSort0(object[] dest, int low, int high) {
		Contract.Requires (dest != null);
		// Contract.Requires (TArrays.within(dest, low, high)); inline for better analysis
        Contract.Requires(0 <= low && low <= high && high <= dest.Length);
		Contract.Requires (TArrays.nonnull(dest, low, high));
		// Skip assignable annotation
		Contract.Ensures (TArrays.sorted (dest, low, high));

		if (dest.Length <= 1)
			return;

		if (low == high) {
			return;
		}

		int ic = low + 1;

		while (ic < high) {
			object new_value = dest[ic];
			dest[ic] = dest[ic - 1];
			int jc = ic;
			while (jc > low && dest[jc - 1].GetHashCode() > new_value.GetHashCode()) {
				dest[jc] = dest[jc - 1];
				jc--;
			}
			dest[jc] = new_value;
			ic++;
		}
	}

    [ContractVerification(false)]
	private class PartitionIndexes {
		internal int a;
		internal int b;
		internal int c;
		internal int d;
	}

	[Pure]
    [ContractVerification(false)]
	public static bool equals0(int[] a1, int[] a2) {
        // Post-condition split up in two parts for readability, but counted as one post-condition.
		Contract.Ensures(!Contract.Result<Boolean>() ||  ((a1 == a2) || ((a1 != null) && (a2 != null) && TArrays.eq(a1, 0, a1.Length, a2, 0, a2.Length))));
		Contract.Ensures( Contract.Result<Boolean>() || !((a1 == a2) || ((a1 != null) && (a2 != null) && TArrays.eq(a1, 0, a1.Length, a2, 0, a2.Length))));
		if (a1 == a2) {
			return true;
		}
		if (a1 == null || a2 == null) {
			return false;
		}
        int length = a1.Length;
		if (length != a2.Length) {
			return false;
		}
		for (int ic = 0; ic < length; ic++) {
			if (a1[ic] != a2[ic]) {
				return false;
			}
		}
		return true;
	}

	[Pure]
    [ContractVerification(false)]
	public static int hashCode0(int[] a) {
		Contract.Ensures((a != null) || (Contract.Result<int>() == 0));
		if (a == null) {
			return 0;
		}
		int result = 1;
		for (int ic = 0; ic < a.Length; ic++) {
			int element = a[ic];
			result = 31 * result + element;
		}
		return result;
	}

	[Pure]
    [ContractVerification(false)]
	public static int hashCodeInteger(object[] a) {
		Contract.Ensures((a != null) || (Contract.Result<int>() == 0));
		if (a == null) {
			return 0;
		}
		int result = 1;
		for (int ic = 0; ic < a.Length; ic++) {
			object element = a[ic];
			result = 31 * result + (element == null ? 0 : element.GetHashCode());
		}
		return result;
	}

    [ContractVerification(false)]
	public static void fillInteger0(object[] a, object val) {
		Contract.Requires(a !=  null);
		// Skip assignable annotation.
		Contract.Ensures(val == null || TArrays.eq(a, 0, a.Length, val));
		// Contract.Ensures(val == null || TArrays.nonnull(a, 0, a.Length)); inlined for better analysis
        Contract.Ensures(val == null || Contract.ForAll(0, a.Length, i => a[i] != null));
		Contract.Ensures(val != null || TArrays.eqnull(a, 0, a.Length));
		for (int ic = 0; ic < a.Length; ic++) {
			a[ic] = val;
		}
	}

    [ContractVerification(false)]
	public static void fillInteger1(object[] a, int fromIndex, int toIndex, object val) {
		Contract.Requires(a !=  null);
		// Contract.Requires(TArrays.within(a, fromIndex, toIndex)); inline for better analysis
        Contract.Requires(0 <= fromIndex && fromIndex <= toIndex && toIndex <= a.Length);
		// Skip assignable annotation
		Contract.Ensures(Contract.ForAll(0, fromIndex, i => a[i] == Contract.OldValue(a[i])));
		Contract.Ensures(Contract.ForAll(toIndex, a.Length, i => a[i] == Contract.OldValue(a[i])));
		Contract.Ensures(val == null || TArrays.nonnull (a, fromIndex, toIndex));
		Contract.Ensures(val == null || TArrays.eq (a, fromIndex, toIndex, val));
		Contract.Ensures(val != null || TArrays.eqnull (a, fromIndex, toIndex));

		rangeCheck(a.Length, fromIndex, toIndex);
		for (int ic = fromIndex; ic < toIndex; ic++) {
			a[ic] = val;
		}
	}
}
} 

