using System;
using System.Diagnostics.Contracts;

namespace CaseStudy
{

public class ArrayList
{
	private int modCount;

	public object[] elementData;

	public int size;

	[ContractInvariantMethod]
    [ContractVerification(false)]
	private void ObjectInvariant() {
		Contract.Invariant(elementData != null);
		Contract.Invariant(Contract.ForAll(size, elementData.Length, i => elementData[i] == null));
		// Skipped owner condition for elementData.
		Contract.Invariant(0 <= size && size <= elementData.Length);
	}

    [ContractVerification(false)]
	public ArrayList(int initialCapacity) {
		if (initialCapacity < 0)
			throw new ArgumentException();
		this.elementData = new object[initialCapacity];
	}

	// Omitted other constructors.

	[Pure]
    [ContractVerification(false)]
	public int indexOf(object value) {
        // Post-condition is split up for readability, but counted as one.
        // Inlining has not effect.
		Contract.Ensures(Contract.Result<int>() != -1 || !TLists.In(elementData, 0, size, value));
		Contract.Ensures(Contract.Result<int>() == -1 ||  TLists.In(elementData, 0, size, value));
		Contract.Ensures(Contract.Result<int>() == -1 || 0 <= Contract.Result<int>() && Contract.Result<int>() < size);
		Contract.Ensures(Contract.Result<int>() == -1 || elementData[Contract.Result<int>()] == value && !TLists.In(elementData, 0, Contract.Result<int>(),value));
		if (value == null) {
			for (int ic = 0; ic < size; ic++) {
				if (elementData[ic] == null) {
					return ic;
				}
			}
		} else {
			for (int ic = 0; ic < size; ic++) {
				if (value == elementData[ic]) {
					return ic;
				}
			}
		}
		return -1;
	}

	[Pure]
    [ContractVerification(false)]
	public int lastIndexOf(object value) {
        // Post-condition is split up for readability, but counted as one.
        // Inlining has not effect.
		Contract.Ensures(Contract.Result<int>() != -1 || !TLists.In(elementData, 0, size, value));
		Contract.Ensures(Contract.Result<int>() == -1 ||  TLists.In(elementData, 0, size, value));
		Contract.Ensures(Contract.Result<int>() == -1 || 0 <= Contract.Result<int>() && Contract.Result<int>() < size);
		Contract.Ensures(Contract.Result<int>() == -1 || elementData[Contract.Result<int>()] == value && !TLists.In(elementData, Contract.Result<int>() + 1, size, value));
		if (value == null) {
			for (int ic = size - 1; ic >= 0; ic--) {
				if (elementData[ic] == null) {
					return ic;
				}
			}
		} else {
			for (int ic = size - 1; ic >= 0; ic--) {
				if (value == elementData[ic]) {
					return ic;
				}
			}
		}
		return -1;
	}

    [ContractVerification(false)]
	private void fastRemove(int index) {
		Contract.Requires(0 <= index && index < size);
		// Skip assignable annotation.
		Contract.Ensures(size == Contract.OldValue(size) - 1);
		Contract.Ensures(Contract.ForAll(0, index, i => elementData[i] == Contract.OldValue(elementData[i])));
		Contract.Ensures(Contract.ForAll(index, size, i => elementData[i] == Contract.OldValue(elementData[i + 1])));
		modCount++;
		int numMoved = size - index - 1;
		if (numMoved > 0)
			SystemUtil.arraycopyInteger(elementData, index + 1, elementData, index, numMoved);
		elementData[--size] = null;
	}

    [ContractVerification(false)]
	public bool remove0(object value) {
		// Skip assignable annotation.
		Contract.Ensures( Contract.Result<bool>() || size == Contract.OldValue(size));
		// Contract.Ensures( Contract.Result<bool>() || Contract.OldValue(!TLists.In(elementData, 0, size, value))); inlined for better analysis
        Contract.Ensures( Contract.Result<bool>() || !Contract.Exists(0, Contract.OldValue(size), i => Contract.OldValue(elementData[i] == value)));
		Contract.Ensures( Contract.Result<bool>() || Contract.ForAll(0, size, i => Contract.OldValue(elementData[i] == elementData[i])));
		Contract.Ensures(!Contract.Result<bool>() || size == Contract.OldValue(size) - 1);
		// Contract.Ensures(!Contract.Result<bool>() || Contract.OldValue(TLists.In(elementData, 0, size, value))); inlined for better analysis
        Contract.Ensures(!Contract.Result<bool>() || Contract.Exists(0, Contract.OldValue(size), i => Contract.OldValue(elementData[i] == value)));
		int index = 0;
		if (value == null) {
			for (; index < size; index++) {
				if (elementData[index] == null) {
					break;
				}
			}
		} else {
			for (; index < size; index++) {
				if (value == elementData[index]) {
					break;
				}
			}
		}
		if (index < size) {
			fastRemove(index);
			return true;
		} else {
			return false;
		}
	}

    [ContractVerification(false)]
	public void clear() {
		// Skipped assignable condition.
		Contract.Ensures(size == 0);
		Contract.Ensures(TLists.eq(elementData, 0, Contract.OldValue(size), null));
        Contract.Ensures(TLists.eq(elementData, Contract.OldValue(size), elementData.Length, null)); // inlining has no effect
		modCount++;
		for (int ic = 0; ic < size; ic++) {
			elementData[ic] = null;
		}

		size = 0;
	}

	// Skip batchRemove

	// Skip contains

    [ContractVerification(false)]
	public object[] toArray() {
		// Skip assignable annotation
		Contract.Ensures(Contract.ForAll(0, elementData.Length, i => elementData[i] == Contract.OldValue(elementData[i])));
		Contract.Ensures(Contract.Result<object[]>() != null);
		Contract.Ensures(Contract.Result<object[]>().Length == size);
		Contract.Ensures(Contract.ForAll(0, Contract.Result<object[]>().Length, i => Contract.Result<object[]>()[i] == elementData[i]));
		// Skip owner annotation

		object[] ret_val = Arrays.copyOf(elementData, size);
		return ret_val;
	}

	[Pure]
    [ContractVerification(false)]
	public int Size() {
		Contract.Ensures(Contract.Result<int>() == size);
		return size;
	}

    [Pure]
    [ContractVerification(false)]
	public object getIndex(int index) {
		Contract.Requires (0 <= index && index < size);
		Contract.Ensures (Contract.Result<object>() == elementData[index]);
		// Skip exceptional behaviour
		if (index < 0 || index >= size) {
			throw new IndexOutOfRangeException();
		}

		object ret_value = elementData[index];
		return ret_value;
	}

	// Skip isEmpty and cloneMe

    [ContractVerification(false)]
	public object setIndex(int index, object new_value) {
		Contract.Requires (0 <= index && index < size);
		// Skip assignable annotation.
		Contract.Ensures (elementData [index] == new_value);
		Contract.Ensures (Contract.ForAll(0, size, i => i == index || elementData[i] == Contract.OldValue(elementData[i])));
		Contract.Ensures (Contract.Result<object>() == Contract.OldValue(elementData[index]));
		// Skip exceptional behaviour.
		if (index < 0 || index >= size) {
			throw new IndexOutOfRangeException();
		}

		object old_value = elementData[index];
		elementData[index] = new_value;
		return old_value;
	}

	// Skip trimToSize

    [ContractVerification(false)]
	public void ensureCapacity(int minCapacity) {
		modCount++;
		int oldCapacity = elementData.Length;
		if (minCapacity > oldCapacity) {
			object[] oldData = elementData;
			int newCapacity = (oldCapacity * 3) / 2 + 1;
			if (newCapacity < minCapacity)
				newCapacity = minCapacity;
			elementData = Arrays.copyOf(elementData, newCapacity);
		}
	}

	// Skip methods

    [ContractVerification(false)]
	public bool add(object new_value) {
		// Skip assignable annotation
		Contract.Ensures(size == 1 + Contract.OldValue (size));
		Contract.Ensures(Contract.ForAll(0, Contract.OldValue(size), i => elementData[i] == Contract.OldValue(elementData[i])));
		Contract.Ensures(elementData[Contract.OldValue(size)] == new_value);
		ensureCapacity(size + 1);
		elementData[size++] = new_value;
		return true;
	}

	[Pure]
    [ContractVerification(false)]
	public object[] getElementData() {
		Contract.Ensures (Contract.Result<object[]>() == elementData);
		object[] copyOf = Arrays.copyOf(elementData, elementData.Length);
		return copyOf;
	}

}
}